#include "gpib.h"
#include "formatio.h"
#include "tga1240.h"

/*= TTi TGA1240 40Ms/s Arbitrary Generator ================================*/
/* LabWindows 5.0 Multi-Instrument Driver                                  */
/* Original Release: March 2000                                            */
/* By: John Tothill, TTi Ltd.                                              */
/* Modification History: None                                              */
/*=========================================================================*/

/*= INSTRUMENT TABLE ======================================================*/
/* address array: contains the GPIB addresses of opened instruments.       */
/* bd array: contains the device descriptors returned by OpenDev.          */
/* instr_cnt: contains the number of instruments open of this model type.  */
/* tga1240_err: the error variable for the instrument module               */
/*=========================================================================*/
static int address[tga1240_MAX_INSTR + 1];
static int bd[tga1240_MAX_INSTR + 1];
static int instr_cnt;
static int tga1240_err;

/*= INTERNAL DATA =========================================================*/
/* cmd is a buffer for GPIB I/O strings */
static char cmd[2000];
static int max_chan[tga1240_MAX_INSTR + 1];
#define ARB_BUF_LEN	66000
static char arb_buf[ARB_BUF_LEN]; /* buffer to take a 16k point waveform max */

/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS ===================================*/
static char *func[14];
static char *freq[4];
static char *zload[3];
static char *opsw[4];
static char *trig_src[5];
static char *set_status[3];
static char *set_store[2];
static char *get_status[6];
static char *sync_op[8];
static char *filt[5];
static char *trig_out[5];
static char *hold_stat[5];
static char *swp_type[3];
static char *swp_dirn[4];
static char *swp_spacing[2];
static char *swp_sync[2];
static char *seq_stp[3];
static char *mod_src[3];
static char *mod_type[2];
static char *sum_atten[6];
static char *lock_mode[4];

/*=========================================================================*/

/*= UTILITY ROUTINES ======================================================*/
int tga1240_open_instr (int);
int tga1240_close_instr (int);
int tga1240_invalid_integer_range (int, int, int, int);
int tga1240_invalid_longint_range (long, long, long, int);
int tga1240_invalid_real_range (double, double, double, int);
int tga1240_device_closed (int);
int tga1240_read_data (int, char *, int);
int tga1240_write_data (int, char *, int);
int tga1240_read_data_file (int, char *);
int tga1240_write_data_file (int, char *);
int tga1240_check_exec_err(int);
void tga1240_setup_arrays (void);
/*=========================================================================*/
/* Function: Initialize                                                    */
/* Purpose:  This function opens the instrument, queries the instrument    */
/*           for its ID, and initializes the instrument to a known state.  */
/*=========================================================================*/
int __stdcall tga1240_init (addr, id_query, rest, instrID)
int addr;
int id_query;
int rest;
int *instrID;
{
int  ID;

    if (tga1240_invalid_integer_range (addr, 0, 30, -1) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (id_query, 0, 1, -2) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (rest, 0, 1, -3) != 0)
        return tga1240_err;

    ID = tga1240_open_instr (addr);
    if (ID <= 0)
        return tga1240_err;

	ibclr(bd[ID]);
    if (id_query)
        {
        if (tga1240_write_data (ID, "*IDN?\012", 6) != 0)
            {
            tga1240_close_instr (ID);
            return tga1240_err;
            }
        if (tga1240_read_data (ID, cmd, 50) != 0)
            {
            tga1240_close_instr (ID);
            return tga1240_err;
            }

        Scan (cmd, "THURLBY THANDAR,TGA124%d",&max_chan[ID]);
        if (NumFmtdBytes () != 23)
            {
            tga1240_err = 223;
            tga1240_close_instr (ID);
            return tga1240_err;
            }
        }
    max_chan[ID]--;
    if (rest)
        {
        if (tga1240_write_data (ID, "*RST\012", 5) != 0)
            {
            tga1240_close_instr (ID);
            return tga1240_err;
            }
         }

    if (tga1240_write_data (ID, "BEEPMODE ERROR\012", 15) != 0)
        return tga1240_err;

    tga1240_setup_arrays ();
    *instrID = ID;
    return tga1240_err;
}
/*=========================================================================*/
/* Function: tga1240_config_wfm                                            */
/* Purpose:  Configure main generator waveform, frequency, output level and*/
/*           output state                                                  */
/*=========================================================================*/
int __stdcall tga1240_config_wfm (int instrumentID, int function,
                                double frequencyPeriod, double amplitudeVpp,
                                double DCOffsetV, int sense, int load,
                                int output, char arbwfmname[], int freqPer, int chan)
{
    if (tga1240_invalid_integer_range (instrumentID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
	if (tga1240_device_closed(instrumentID))
	    return tga1240_err;
    if (tga1240_invalid_integer_range (chan, 0, max_chan[instrumentID], -10) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (freqPer, 0, 3, -6) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (function, 0, 13, -2) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (output, 0, 1, -3) != 0)
        return tga1240_err;
    if (tga1240_invalid_real_range (frequencyPeriod, 1e-4, 40e6, -4) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (sense, 0, 2, -9) != 0)
        return tga1240_err;
    if (tga1240_invalid_real_range (DCOffsetV, -10.0, 10.0, -5) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (load, 0, 2, -8) != 0)
        return tga1240_err;
    if (tga1240_device_closed (instrumentID))
        return tga1240_err;
    Fmt(cmd,"SETUPCH %d\012",chan+1);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
    if (function != 12) /* not an arb */
        Fmt(cmd,"WAVE %s\012",func[function]);
    else
        Fmt(cmd,"ARB %s\012",arbwfmname);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
    Fmt(cmd,"%s %f;ZLOAD %s;AMPUNIT VPP;AMPL %f;DCOFFS %f;OUTPUT %s;OUTPUT %s\012"
            ,freq[freqPer],frequencyPeriod,zload[load],amplitudeVpp
            ,DCOffsetV,opsw[sense+2],opsw[output]);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
    tga1240_check_exec_err(instrumentID);    
    return tga1240_err;
}
/*=========================================================================*/
/* Function: tga1240_config_pulse                                          */
/* Purpose:  Configure the parameters for a pulse waveform                 */
/*=========================================================================*/
int __stdcall tga1240_config_pulse (int instrumentID, int channel,
                                  double pulsePeriod, double pulseWidth,
                                  double pulseDelay)
{
    if (tga1240_invalid_integer_range (instrumentID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
	if (tga1240_device_closed(instrumentID))
	    return tga1240_err;
    if (tga1240_invalid_integer_range (channel, 0, max_chan[instrumentID], -10) != 0)
        return tga1240_err;
    if (tga1240_invalid_real_range (pulsePeriod, 100e-9, 100, -2) != 0)
        return tga1240_err;
    if (tga1240_invalid_real_range (pulseWidth, 25e-9, 99.99, -3) != 0)
        return tga1240_err;
    if (tga1240_invalid_real_range (pulseDelay, -99.99, 99.99, -4) != 0)
        return tga1240_err;
    if (tga1240_device_closed (instrumentID))
        return tga1240_err;
    Fmt(cmd,"SETUPCH %d;PULSPER %f;PULSWID %f;PULSDLY %f\012"
    			,channel+1,pulsePeriod,pulseWidth,pulseDelay);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
    tga1240_check_exec_err(instrumentID);    
    return tga1240_err;
}
/*=========================================================================*/
/* Function: tga1240_config_pulse_trn                                      */
/* Purpose:  Configure the parameters for a pulse train waveform           */
/*=========================================================================*/
int __stdcall tga1240_config_pulse_trn (int instrumentID, int channel,
                                      int pulseNumber, double pulseTrainPeriod,
                                      double baseline, double pulseLevel,
                                      double pulseWidth, double pulseDelay,
                                      int Number_of_Pulses, int Make)
{
    if (tga1240_invalid_integer_range (instrumentID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
	if (tga1240_device_closed(instrumentID))
	    return tga1240_err;
    if (tga1240_invalid_integer_range (channel, 0, max_chan[instrumentID], -10) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (pulseNumber, 1, 10, -5) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (Number_of_Pulses, 1, 10, -8) != 0)
        return tga1240_err;
    if (tga1240_invalid_real_range (pulseTrainPeriod, 100e-9, 100, -2) != 0)
        return tga1240_err;
    if (tga1240_invalid_real_range (baseline, -5, 5, -6) != 0)
        return tga1240_err;
    if (tga1240_invalid_real_range (pulseLevel, -5, 5, -7) != 0)
        return tga1240_err;
    if (tga1240_invalid_real_range (pulseWidth, 25e-9, 99.99, -3) != 0)
        return tga1240_err;
    if (tga1240_invalid_real_range (pulseDelay, -99.99, 99.99, -4) != 0)
        return tga1240_err;
    if (tga1240_device_closed (instrumentID))
        return tga1240_err;
    if (tga1240_invalid_integer_range (Make, 0, 1, -9) != 0)
        return tga1240_err;
    Fmt(cmd,"SETUPCH %d;PULTRNLEN %d;PULTRNPER %f;PULTRNBASE %f\012"
    			,channel+1,Number_of_Pulses,pulseTrainPeriod,baseline);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
    Fmt (cmd,"PULTRNWID %d,%f;PULTRNDLY %d,%f;PULTRNLEV %d,%f\012"
    			  ,pulseNumber,pulseWidth,pulseNumber,pulseDelay,pulseNumber,pulseLevel);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
	if (Make != 0)
	    {
	    Fmt (cmd,"PULTRNMAKE\012");
    	if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        	return tga1240_err;
	    }
    tga1240_check_exec_err(instrumentID);    
    return tga1240_err;
    
}
/*=========================================================================*/
/* Function: tga1240_arb_info                                              */
/* Purpose:  Get the length of the specified arb if it exists              */
/*=========================================================================*/
int __stdcall tga1240_arb_info (int instrumentID, char arbName[], int *length)
{
    if (tga1240_invalid_integer_range (instrumentID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
	if (tga1240_device_closed(instrumentID))
	    return tga1240_err;
    Fmt (cmd,"ARBLEN? %s\012",arbName);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
    if (tga1240_read_data (instrumentID, cmd, 50) != 0)
        {
        tga1240_close_instr (instrumentID);
        return tga1240_err;
        }
    Scan (cmd, "%d",length);
	tga1240_check_exec_err(instrumentID);    
    return tga1240_err;
}
/*=========================================================================*/
/* Function: tga1240_arb_create                                            */
/* Purpose:  Create a new arb as a blank waveform or a copy of an existing */
/*           waveform.                                                     */
/*=========================================================================*/
int __stdcall tga1240_arb_create ( int instrumentID, char newarbname[],int create,
                                int length, int waveformtocopy,
                                char arbitraryname[])
{
    if (tga1240_invalid_integer_range (instrumentID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
	if (tga1240_device_closed(instrumentID))
	    return tga1240_err;
    if (tga1240_invalid_integer_range (create, 0, 1, -2) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (length, 4, 65536, -3) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (waveformtocopy, 0, 10, -4) != 0)
        return tga1240_err;
    Fmt (cmd,"ARBCREATE %s,%d\012",newarbname,length);
    ibtmo(bd[instrumentID],T300s);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
		{
		ibtmo(bd[instrumentID],T30s);
        return tga1240_err;
        }
	tga1240_check_exec_err(instrumentID);    
	if (tga1240_err)
		{
		ibtmo(bd[instrumentID],T30s);
	    return tga1240_err;
	    }
	if (create == 1)
	    {
	    if (waveformtocopy == 10) /* arb */
	    	{
    		Fmt (cmd,"ARBINSARB %s,%s,0,%d\012",newarbname,arbitraryname,length-1);
    		if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
				{
				ibtmo(bd[instrumentID],T30s);
        		return tga1240_err;
        		}
	    	}
	    else    /* standard */
	    	{
    		Fmt (cmd,"ARBINSSTD %s,%s,0,%d\012",newarbname,func[waveformtocopy]
    												,length-1);
    		if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
				{
				ibtmo(bd[instrumentID],T30s);
        		return tga1240_err;
        		}
	    	}
	    }
	tga1240_check_exec_err(instrumentID);    
    ibtmo(bd[instrumentID],T30s);
    return tga1240_err;
}
/*=========================================================================*/
/* Function: tga1240_arb_modify                                            */
/* Purpose:  Delete, Resize or Rename an existing arb waveform             */
/*=========================================================================*/
int __stdcall tga1240_arb_modify (int instrumentID, char arbname[], int operation,
                                int newlength, char newarbname[])
{
    if (tga1240_invalid_integer_range (instrumentID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
	if (tga1240_device_closed(instrumentID))
	    return tga1240_err;
    if (tga1240_invalid_integer_range (operation, 0, 3, -2) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (newlength, 4, 65536, -3) != 0)
        return tga1240_err;
	ibtmo(bd[instrumentID],T300s);
    switch (operation)
    {
    case 0: /* delete */
        Fmt (cmd,"ARBDELETE %s\012",arbname);
		break;
    case 1: /* rename */
        Fmt (cmd,"ARBRENAME %s,%s\012",arbname,newarbname);
		break;
    case 2: /* resize */
        Fmt (cmd,"ARBRESIZE %s,%d\012",arbname,newlength);
		break;
    case 3: /* channel del */
        Fmt (cmd,"ARBCLR %s\012",arbname);
		break;
    }
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
		{
		ibtmo(bd[instrumentID],T30s);
        return tga1240_err;
        }
	tga1240_check_exec_err(instrumentID);    
	ibtmo(bd[instrumentID],T30s);
	return tga1240_err;
}
/*=========================================================================*/
/* Function: tga1240_arb_edit                                              */
/* Purpose:  Perform the specified edit operation on the arb "arbname"     */
/*=========================================================================*/
int __stdcall tga1240_arb_edit (int instrumentID, char arbname[], int operation,
                              int firstpointidx, int firstpointvalue,
                              int secondpointidx, int secondpointvalue,
                              int destinationidx, float amplitudemultiplier,
                              int offsetvalue, int waveformtoinsert, char insertarb[])
{
    if (tga1240_invalid_integer_range (instrumentID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
	if (tga1240_device_closed(instrumentID))
	    return tga1240_err;
    if (tga1240_invalid_integer_range (operation, 0, 8, -2) != 0)
        return tga1240_err;
	ibtmo(bd[instrumentID],T300s);
    switch (operation)
    {
    case 0: /* point edit */
        Fmt (cmd,"ARBPOINT %s,%d,%d\012",arbname,firstpointidx,firstpointvalue);
		break;
    case 1: /* line draw */
        Fmt (cmd,"ARBLINE %s,%d,%d,%d,%d\012",arbname,firstpointidx,firstpointvalue
        												,secondpointidx,secondpointvalue);
		break;
    case 2: /* wave insert */
    	if (tga1240_invalid_integer_range (waveformtoinsert, 0, 10, -3) != 0)
			{
			ibtmo(bd[instrumentID],T30s);
        	return tga1240_err;
        	}
        if (waveformtoinsert == 10) /* arb */
            Fmt (cmd,"ARBEDLMTS 0,0;ARBINSARB %s,%s,%d,%d\012"
            							,arbname,insertarb,firstpointidx,secondpointidx);
		else
            Fmt (cmd,"ARBINSSTD %s,%s,%d,%d\012"
            					,arbname,func[waveformtoinsert],firstpointidx,secondpointidx);
		break;
    case 3: /* block copy */
        Fmt (cmd,"ARBCOPY %s,%d,%d,%d\012",arbname,firstpointidx,secondpointidx,destinationidx);
		break;
    case 4: /* amplitude */
        Fmt (cmd,"ARBAMPL %s,%d,%d,%f\012",arbname,firstpointidx,secondpointidx,amplitudemultiplier);
		break;
    case 5: /* offset */
        Fmt (cmd,"ARBOFFSET %s,%d,%d,%d\012",arbname,firstpointidx,secondpointidx,offsetvalue);
		break;
    case 6: /* wave invert */
        Fmt (cmd,"ARBINVERT %s,%d,%d\012",arbname,firstpointidx,secondpointidx);
 		break;
    case 7: /* set posn mkr */
        Fmt (cmd,"POSNMKRSET %s,%d\012",arbname,firstpointidx);
		break;
    case 8: /* clear all markers */
        Fmt (cmd,"POSNMKRCLR %s\012",arbname);
		break;
    }
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
		{
		ibtmo(bd[instrumentID],T30s);
        return tga1240_err;
        }
	tga1240_check_exec_err(instrumentID);    
	ibtmo(bd[instrumentID],T30s);
	return tga1240_err;
}
/*=========================================================================*/
/* Function: tga1240_setup_mode                                            */
/* Purpose:  Set the mode and the trigger source. Also set burst count     */
/*           and stop/start phase angle									   */
/*=========================================================================*/
int __stdcall tga1240_setup_mode (int instrumentID, int channel, int mode,
                                int triggersource, double internalperiod,
                                int burstcount, int triggerslope,
                                float phaseangle)
{
    if (tga1240_invalid_integer_range (instrumentID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
	if (tga1240_device_closed(instrumentID))
	    return tga1240_err;
    if (tga1240_invalid_integer_range (channel, 0, max_chan[instrumentID], -10) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (mode, 0, 4, -2) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (triggersource, 0, 4, -3) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (burstcount, 1, 1048575, -5) != 0)
        return tga1240_err;
	Fmt (cmd,"SETUPCH %d;TRIGIN %s;TRIGIN %s;BSTCNT %d;PHASE %f;TRIGPER %f\012"
	 							,channel+1
								,trig_src[triggersource],triggerslope == 0 ? "POS" : "NEG"
								,burstcount,phaseangle,internalperiod);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
	if (tga1240_check_exec_err(instrumentID) != 0)
	    return tga1240_err;;    
    switch (mode)
    {
    case 0:
        Fmt (cmd,"MODE CONT\012");
		break;
    case 1: 
        Fmt (cmd,"MODE GATE\012");
		break;
    case 2:
        Fmt (cmd,"MODE TRIG\012");
	    break;
    case 3:
        Fmt (cmd,"MODE SWEEP\012");
		break;
    case 4:
        Fmt (cmd,"MODE TONE\012");
		break;
    }
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
	tga1240_check_exec_err(instrumentID);    
	return tga1240_err;
}
/*=========================================================================*/
/* Function: tga1240_setup_connector                                       */
/* Purpose:  Set the various i/o options of the channel or instrument      */
/*=========================================================================*/
int __stdcall tga1240_setup_conector (int instrumentID, int channel, int output,
                                    int syncout, int syncoutsignal,
                                    int outputfilter, int triggerout, int hold)
{
    if (tga1240_invalid_integer_range (instrumentID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
	if (tga1240_device_closed(instrumentID))
	    return tga1240_err;
    if (tga1240_invalid_integer_range (channel, 0, max_chan[instrumentID], -10) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (output, 0, 1, -2) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (syncout, 0, 1, -3) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (syncoutsignal, 0, 7, -4) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (outputfilter, 0, 4, -5) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (triggerout, 0, 4, -6) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (hold, 0, 3, -7) != 0)
        return tga1240_err;
	Fmt (cmd,"SETUPCH %d;OUTPUT %s;SYNCOUT %s;SYNCOUT %s;FILTER %s;TRIGOUT %s;HOLD %s\012"
							,channel+1,opsw[output],opsw[syncout],sync_op[syncoutsignal]
							,filt[outputfilter],trig_out[triggerout],hold_stat[hold]);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
	tga1240_check_exec_err(instrumentID);
	return tga1240_err;;    
}
/*===========================================================================*/
/* Function: tga1240_setup_sweep                                             */
/* Purpose:  Set the sweep parameters and enable sweep for selected channels */
/*===========================================================================*/
int __stdcall tga1240_setup_sweep (int instrumentID, int channel, int sweeptype,
                                 int direction, int spacing,
                                 float startfrequency, float stopfrequency,
                                 int sync, float duration, float markerfrequency,
                                 int control)
{
    if (tga1240_invalid_integer_range (instrumentID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
	if (tga1240_device_closed(instrumentID))
	    return tga1240_err;
    if (tga1240_invalid_integer_range (channel, 0, max_chan[instrumentID], -10) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (control, 0, 1, -2) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (sweeptype, 0, 12, -3) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (direction, 0, 3, -4) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (spacing, 0, 1, -5) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (sync, 0, 1, -6) != 0)
        return tga1240_err;
	/* stop sweep while paraneters are set up */
    Fmt (cmd,"MODE CONT\012");
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
	Fmt (cmd,"SETUPCH %d;SWPTYPE %s;SWPDIRN %s;SWPSPACING %s;SWPSTARTFRQ %f;SWPSTOPFRQ %f;SWPSYNC %s;SWPTIME %f;SWPMKR %f\012"
					,channel+1,swp_type[sweeptype],swp_dirn[direction],swp_spacing[spacing]
					,startfrequency,stopfrequency,swp_sync[sync],duration,markerfrequency);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
    if (tga1240_check_exec_err(instrumentID) != 0)
        return tga1240_err;
    if (control == 0)   /* off */
        Fmt (cmd,"MODE CONT\012");
    else
        Fmt (cmd,"MODE SWEEP\012");
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
	tga1240_check_exec_err(instrumentID);
	return tga1240_err;;    
}
/*=========================================================================*/
/* Function: tga1240_setup_seq                                             */
/* Purpose:  Set the waveform sequence parameters for the selected step    */
/*=========================================================================*/
int __stdcall tga1240_setup_seq (int instrumentID, int channel, int segmentno,
                               char arbwfmname[], int status, int count,
                               int stepon)
{
    if (tga1240_invalid_integer_range (instrumentID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
	if (tga1240_device_closed(instrumentID))
	    return tga1240_err;
    if (tga1240_invalid_integer_range (channel, 0, max_chan[instrumentID], -10) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (segmentno, 1, 16, -2) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (status, 0, 1, -3) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (stepon, 0, 2, -4) != 0)
        return tga1240_err;
	Fmt (cmd,"SETUPCH %d;SEQWFM %d,%s;SEQSTEP %d,%s;SEQCNT %d,%d\012"
					,channel+1,segmentno,arbwfmname,segmentno,seq_stp[stepon]
					,segmentno,count);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
	if (tga1240_check_exec_err(instrumentID) !=0 )
	    return tga1240_err;
	if (segmentno > 1)
	    {
	    Fmt (cmd,"SEQSEG %d,%s\012",segmentno,swp_sync[status]);
        if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
            return tga1240_err;
	    }
	tga1240_check_exec_err(instrumentID);
	return tga1240_err;;    
}
/*=========================================================================*/
/* Function: tga1240_setup_tone                                            */
/* Purpose:  Set the tone frequency list and tone type                     */
/*=========================================================================*/
int __stdcall tga1240_setup_tone (int instrumentID, int channel, int tonenumber,
                                float tonefrequency, int tonetype)
{
    if (tga1240_invalid_integer_range (instrumentID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
	if (tga1240_device_closed(instrumentID))
	    return tga1240_err;
    if (tga1240_invalid_integer_range (channel, 0, max_chan[instrumentID], -10) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (tonetype, 0, 2, -2) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (tonenumber, 1, 16, -3) != 0)
        return tga1240_err;
	if (tonefrequency == -1) /* set end of list */
		Fmt (cmd,"SETUPCH %d;TONEEND %d\012",channel+1,tonenumber);
	else
	    Fmt (cmd,"SETUPCH %d;TONEFREQ %d,%f,%d\012"
	 							,channel+1,tonenumber,tonefrequency,tonetype);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
	tga1240_check_exec_err(instrumentID);
	return tga1240_err;;    
}
/*=========================================================================*/
/* Function: tga1240_setup_mod                                             */
/* Purpose:  Set modulation for the specified channel                      */
/*=========================================================================*/
int __stdcall tga1240_setup_mod (int instrumentID, int channel, int source,
                               int type, float SCMlevel, float AMdepth)
{
    if (tga1240_invalid_integer_range (instrumentID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
	if (tga1240_device_closed(instrumentID))
	    return tga1240_err;
    if (tga1240_invalid_integer_range (channel, 0, max_chan[instrumentID], -10) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (source, 0, 2, -2) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (type, 0, 1, -3) != 0)
        return tga1240_err;
    Fmt (cmd,"SETUPCH %d;MOD %s;MODTYPE %s;AMDEPTH %f;SCMLEVEL %f\012"
	 							,channel+1,mod_src[source],mod_type[type],AMdepth,SCMlevel);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
	tga1240_check_exec_err(instrumentID);
	return tga1240_err;;    
}
/*=========================================================================*/
/* Function: tga1240_setup_mod                                             */
/* Purpose:  Set modulation for the specified channel                      */
/*=========================================================================*/
int __stdcall tga1240_setup_sum ( int instrumentID, int channel, int source,
                               float ratio , int atten)
{
    if (tga1240_invalid_integer_range (instrumentID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
	if (tga1240_device_closed(instrumentID))
	    return tga1240_err;
    if (tga1240_invalid_integer_range (channel, 0, max_chan[instrumentID], -10) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (source, 0, 2, -2) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (atten, 0, 5, -3) != 0)
        return tga1240_err;
	if (channel != 0)
    	Fmt (cmd,"SETUPCH %d;SUM %s;SUMRATIO %f;SUMATN %s\012"
	 							,channel+1,mod_src[source],ratio,sum_atten[atten]);
    else
    	Fmt (cmd,"SETUPCH %d;SUM %s;SUMATN %s\012"
	 							,channel+1,mod_src[source],sum_atten[atten]);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
	tga1240_check_exec_err(instrumentID);
	return tga1240_err;;    
}
/*=========================================================================*/
/* Function: tga1240_set_inter                                             */
/* Purpose:  Set inter channel phase locking                               */
/*=========================================================================*/
int __stdcall tga1240_set_inter (int instrumentID, int channel, int mode,
                               float phaseangle, int status)
{
    if (tga1240_invalid_integer_range (instrumentID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
	if (tga1240_device_closed(instrumentID))
	    return tga1240_err;
    if (tga1240_invalid_integer_range (channel, 0, max_chan[instrumentID], -10) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (mode, 0, 3, -2) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (status, 0, 1, -3) != 0)
        return tga1240_err;
   	Fmt (cmd,"SETUPCH %d;LOCKMODE %s;PHASE %f;LOCKSTAT %s\012"
	 							,channel+1,lock_mode[mode],phaseangle,swp_sync[status]);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
	tga1240_check_exec_err(instrumentID);
	return tga1240_err;;    
}
/*=========================================================================*/
/* Function: tga1240_arb_read                                              */
/* Purpose:  Gets data point values from the specified arb waveform        */
/*			 The maximun no of points is 16k							   */
/*=========================================================================*/
int __stdcall tga1240_arb_read (int instrumentID, char arbwfmname[], int start,
                              int end, short wfmdata[])
{
int n;
int num_size,nums;
    if (tga1240_invalid_integer_range (instrumentID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
	if (tga1240_device_closed(instrumentID))
	    return tga1240_err;
    if (start > end)  
        {
        tga1240_err =  -2;
        return tga1240_err;
        }
    Fmt (cmd, "ARBLEN? %s\012",arbwfmname);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
    if (tga1240_read_data (instrumentID, cmd, 20) != 0)
            return tga1240_err;
    Scan (cmd,"%d",&n);
    if (n == 0)
        {
        tga1240_err = -3;
        return tga1240_err;
        }
    if (end >= n)
        {
        tga1240_err = -4;
        return tga1240_err;
        }
	n = end - start + 1;
    if (n > 32768)
    	{
    	tga1240_err = -5;
    	return tga1240_err;
    	}
    Fmt (cmd, "ARBEDLMTS %d,%d;ARBDATA? %s\012",start,end,arbwfmname);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
    if (tga1240_read_data (instrumentID, arb_buf, ARB_BUF_LEN) != 0)
        return tga1240_err;
	if (tga1240_check_exec_err(instrumentID) != 0)
    	return tga1240_err;
    num_size = arb_buf[1]-48;
    if (Scan(arb_buf,"%s>#%d[w1]%d[w*]", &nums, num_size, &n) != 2) 
        {
        tga1240_err = 236;
        return tga1240_err;
        }
    nums += 2;
    n /= 2;
    if (Scan (arb_buf, "%*i[zb2i*o10]>%*i[b2]",n, nums, n, wfmdata) != 1) 
        {
        tga1240_err = 236;
        return tga1240_err;
        }
	return tga1240_err;		
}
/*=========================================================================*/
/* Function: tga1240_arb_write                                             */
/* Purpose:  Sends data point values to the specified arb waveform         */
/*=========================================================================*/
int __stdcall tga1240_arb_write (int instrumentID, char arbwfmname[], int start,
                               int end, short wfmdata[])
{
int nums,n,i;
    if (tga1240_invalid_integer_range (instrumentID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
	if (tga1240_device_closed(instrumentID))
	    return tga1240_err;
    if (start > end)  
        {
        tga1240_err =  -2;
        return tga1240_err;
        }
    Fmt (cmd, "ARBLEN? %s\012",arbwfmname);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
    if (tga1240_read_data (instrumentID, cmd, 20) != 0)
            return tga1240_err;
    Scan (cmd,"%d",&n);
    if (n == 0)
        {
        tga1240_err = -3;
        return tga1240_err;
        }
    if (end >= n)
        {
        tga1240_err = -4;
        return tga1240_err;
        }
	n = end - start + 1;
    if (n > 32768)
    	{
    	tga1240_err = -5;
    	return tga1240_err;
    	}
    Fmt (cmd,"%s<%d",n*2);
    nums = NumFmtdBytes();
    Fmt (arb_buf,"%s<ARBEDLMTS %d,%d;ARBDATA %s,#%d%s",start,end,arbwfmname,nums,cmd);
    nums = NumFmtdBytes();
    for (i=0; i<n; i++)
        {
        arb_buf[2*i+nums] = wfmdata[i] / 256;
        arb_buf[2*i+1+nums] = wfmdata[i] % 256;
        }
	ibtmo(bd[instrumentID],T300s);
    if (tga1240_write_data (instrumentID, arb_buf, nums+2*n) != 0)
        return tga1240_err;
	tga1240_check_exec_err(instrumentID);
	ibtmo(bd[instrumentID],T30s);
	return tga1240_err;		
}
/*=========================================================================*/
/* Function: tga1240_set_status                                            */
/* Purpose:  Set the value in the Service Request Enable, Event Status     */
/*           Enable or Parallel Poll Enable register                       */
/*=========================================================================*/
int __stdcall tga1240_set_status (int instrumentID, int setStatusEnable, int value)
{
    if (tga1240_invalid_integer_range (instrumentID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (setStatusEnable, 0, 2, -2) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (value, 0, 255, -3) != 0)
        return tga1240_err;
    if (tga1240_device_closed (instrumentID))
        return tga1240_err;
    Fmt (cmd, "%s %d\012", set_status[setStatusEnable],value);
    if (tga1240_write_data (instrumentID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
    return tga1240_err;
}
/*=========================================================================*/
/* Function: tga1240_send_trg                                               */
/* Purpose:  Send a trigger command to the TG1010. If MAN/REMOTE is        */
/*           selected as the trigger source for any active mode then that  */
/*           mode will be triggered by this command. If no modes are active*/
/*           then the command will be ignored.                             */
/*=========================================================================*/
int __stdcall tga1240_send_trg (instrID)
int instrID;
{
    if (tga1240_device_closed (instrID))
        return tga1240_err;

    if (tga1240_invalid_integer_range (instrID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;

    Fmt(cmd,"*TRG\012");
    if (tga1240_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
    return tga1240_err;
}
/*=========================================================================*/
/* Function: tga1240_loc_stores                                             */
/* Purpose:  Allows saving to or recalling from the instruments local non- */
/*           volatile stores                                               */
/*=========================================================================*/
int __stdcall tga1240_loc_stores (instrID, op, storeno)
int instrID;
int op;
int storeno;
{
    if (tga1240_invalid_integer_range (instrID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (op, 0, 1, -1) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (storeno, 0, 9, -2) != 0)
        return tga1240_err;
    if (op == 1 && storeno == 0)
        {
        tga1240_err = -2; /* can't write to store 0 */
        return tga1240_err;
        }
    if (tga1240_device_closed (instrID))
        return tga1240_err;

    Fmt (cmd, "%s %d\012", set_store[op],storeno);
    if (tga1240_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;

    return tga1240_err;
}
/*=========================================================================*/
/* Function: tga1240_get status                                             */
/* Purpose:  Reads the value in the selected iEEE 488.2 status register    */
/*=========================================================================*/
int __stdcall tga1240_get_status (instrID, reg, val)
int instrID;
int reg;
int *val;
{
    if (tga1240_invalid_integer_range (instrID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (reg, 0, 5, -2) != 0)
        return tga1240_err;

    if (tga1240_device_closed (instrID))
        return tga1240_err;

    Fmt (cmd, "%s\012", get_status[reg]);
    if (tga1240_write_data (instrID, cmd, NumFmtdBytes ()) != 0)
        return tga1240_err;
    if (tga1240_read_data (instrID, cmd, 50) != 0)
        return tga1240_err;
    if (Scan (cmd, "%d",val) != 1)
        tga1240_err = 236;

    return tga1240_err;
}
/*=========================================================================*/
/* Function: tga1240_snd_cmd                                                */
/* Purpose:  Sends the data in the supplied buffer to the instrument       */
/*           Use for commands not available in this driver - e.g. inc and  */
/*           dec commands.												   */
/*=========================================================================*/
int __stdcall tga1240_snd_cmd (instrID, command)
int instrID;
char *command;
{
char output_buf[200];
    if (tga1240_invalid_integer_range (instrID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;

    if (tga1240_device_closed (instrID))
        return tga1240_err;

    Fmt(output_buf,"%s",command);
    if (tga1240_write_data(instrID,command,NumFmtdBytes()) != 0)
        return tga1240_err;

    return tga1240_err;
}
/*=========================================================================*/
/* Function: tga1240_get_response                                           */
/* Purpose:  Gets a response to a previous query. Used in conjuction with  */
/*           the send command function                                     */
/*=========================================================================*/
int __stdcall tga1240_get_resp (instrID, response, nbytes)
int instrID;
char *response;
int *nbytes;
{
    if (tga1240_invalid_integer_range (instrID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;

    if (tga1240_device_closed (instrID))
        return tga1240_err;

    if (tga1240_read_data(instrID,response,2000))
        return tga1240_err;
    *nbytes = ibcnt;

    return tga1240_err;
}
/*=========================================================================*/
/* Function: tga1240_file_ops                                               */
/* Purpose:  Send or receive files                                         */
/*=========================================================================*/
int __stdcall tga1240_file_ops (instrID, op, fname)
int instrID;
int op;
char *fname;
{
    if (tga1240_invalid_integer_range (instrID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;
    if (tga1240_invalid_integer_range (op, 0, 1, -2) != 0)
        return tga1240_err;

    if (tga1240_device_closed (instrID))
        return tga1240_err;

    if (op == 1) /* get to file from tga1240 */
        {
        if (tga1240_read_data_file(instrID,fname))
            return tga1240_err;
        }
    else /* send from file to tga1240 */
        if (tga1240_write_data_file(instrID,fname))
            return tga1240_err;

    return tga1240_err;
}
/*=========================================================================*/
/* Function: Close                                                         */
/* Purpose:  This function closes the instrument.                          */
/*=========================================================================*/
int __stdcall tga1240_close (instrID)
int instrID;
{
    if (tga1240_invalid_integer_range (instrID, 1, tga1240_MAX_INSTR, -1) != 0)
        return tga1240_err;

    if (tga1240_device_closed (instrID))
        return tga1240_err;

    tga1240_close_instr (instrID);
    return tga1240_err;
}

/*= UTILITY ROUTINES ======================================================*/

/*=========================================================================*/
/* Function: Open Instrument                                               */
/* Purpose:  This function locates and initializes an entry in the         */
/*           Instrument Table and the GPIB device table for the            */
/*           instrument.  If successful, the instrument ID is returned,    */
/*           else a -1 is returned.  The size of the Instrument Table can  */
/*           be changed in the include file by altering the constant       */
/*           tga1240_MAX_INSTR.                                             */
/*=========================================================================*/
int tga1240_open_instr (addr)
int addr;
{
    int i, instrID;

    instrID = 0;

/* Check to see if the instrument is already in the Instrument Table. */

    for (i=1; i<= tga1240_MAX_INSTR; i++)
        if (address[i] == addr)
            {
            instrID = i;
            i = tga1240_MAX_INSTR;
            }

/* If it is not in the instrument table, open an entry for the instrument. */

    if (instrID <= 0)
        for (i=1; i<= tga1240_MAX_INSTR; i++)
            if (address[i] == 0)
                {
                instrID = i;
                i = tga1240_MAX_INSTR;
                }

/* If an entry could not be opened in the Instrument Table, return an error.*/

    if (instrID <= 0)
        {
        tga1240_err = 220;
        return -1;
        }

/*  If the device has not been opened in the GPIB device table (bd[ID] = 0),*/
/*  then open it.                                                           */

    if (bd[instrID] <= 0)
        {
        if (instr_cnt <= 0)
            CloseInstrDevs("tga1240");
        bd[instrID] = OpenDev ("", "tga1240");
        if (bd[instrID] <= 0)
            {
            tga1240_err = 220;
            return -1;
            }
        instr_cnt += 1;
        address[instrID] = addr;
        }

/*  Change the primary address of the device    */

    if (ibpad (bd[instrID], addr) < 0)
        {
        tga1240_err = 233;
        return -1;
        }

    return instrID;
}
/*=========================================================================*/
/* Function: Close Instrument                                              */
/* Purpose:  This function closes the instrument by removing it from the   */
/*           GPIB device table and setting the address and the bd to zero  */
/*           in the Instrument Table.  The return value is equal to the    */
/*           global error variable.                                        */
/*=========================================================================*/
int tga1240_close_instr (instrID)
int instrID;
{
	tga1240_err = 0;
    if (bd[instrID] != 0)
        {
        CloseDev (bd[instrID]);
        bd[instrID] = 0;
        address[instrID] = 0;
        instr_cnt -= 1;
        }
    else
        tga1240_err = 221;

    return tga1240_err;
}
/*=========================================================================*/
/* Function: Invalid Integer Range                                         */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int tga1240_invalid_integer_range (val, min, max, err_code)
int val;
int min;
int max;
int err_code;
{
  if ((val < min) || (val > max))
    {
    tga1240_err = err_code;
    return -1;
    }
  return 0;
}
/*=========================================================================*/
/* Function: Invalid Long Integer Range                                    */
/* Purpose:  This function checks a long integer to see if it lies between */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.  The return value is equal to the     */
/*           global error value.                                           */
/*=========================================================================*/
int tga1240_invalid_longint_range (val, min, max, err_code)
long val;
long min;
long max;
int  err_code;
{
    if (val < min || val > max)
        {
        tga1240_err = err_code;
        return -1;
        }
    return 0;
}
/*=========================================================================*/
/* Function: Invalid Real Range                                            */
/* Purpose:  This function checks a real number to see if it lies between  */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int tga1240_invalid_real_range (val, min, max, err_code)
double val;
double min;
double max;
int err_code;
{
  if ((val < min) || (val > max))
    {
    tga1240_err = err_code;
    return -1;
    }
  return 0;
}
/*=========================================================================*/
/* Function: Device Closed                                                 */
/* Purpose:  This function checks to see if the module has been            */
/*           initialized.  If the device has not been opened, a 1 is       */
/*           returned, 0 otherwise.                                        */
/*=========================================================================*/
int tga1240_device_closed (instrID)
int instrID;
{
    if (bd[instrID] <= 0)
        {
        tga1240_err = 232;
        return -1;
        }
    return 0;
}
/*=========================================================================*/
/* Function: Read Data                                                     */
/* Purpose:  This function reads a buffer of data from the instrument. The */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int tga1240_read_data (instrID, buf, cnt)
int instrID;
char *buf;
int cnt;
{
    if (ibrd(bd[instrID], buf, (long)cnt) <= 0)
        tga1240_err = 231;
    else
        tga1240_err = 0;

    return tga1240_err;
}
/*=========================================================================*/
/* Function: Write Data                                                    */
/* Purpose:  This function writes a buffer of data to the instrument. The  */
/*           return value is equal to the global error variable.           */
/*=========================================================================*/
int tga1240_write_data (instrID, buf, cnt)
int instrID;
char *buf;
int cnt;
{
    if (ibwrt(bd[instrID], buf, (long)cnt) <= 0)
        tga1240_err = 230;
    else
        tga1240_err = 0;

    return tga1240_err;
}
/*=========================================================================*/
/* Function: Read Data File                                                */
/* Purpose:  This function reads a buffer of data from the instrument and  */
/*           stores it to the file specified by "filename".  Filename must */
/*           either be a string, such as "C:\lw\instr\file" or a pointer   */
/*           to such a string.  The return value is equal to the global    */
/*           error variable.                                               */
/*=========================================================================*/
int tga1240_read_data_file (instrID, filename)
int instrID;
char *filename;
{
    if (ibrdf (bd[instrID], filename) <= 0)
        tga1240_err = 229;
    else
        tga1240_err = 0;

    return tga1240_err;
}
/*=========================================================================*/
/* Function: Write Data File                                               */
/* Purpose:  This function writes a buffer of data from the file specified */
/*           by "filename" to the instrument. Filename must either be a    */
/*           string, such as "C:\lw\instr\file" or a pointer to such a     */
/*           string.  The return value is equal to the global error        */
/*           variable.                                                     */
/*=========================================================================*/
int tga1240_write_data_file (instrID, filename)
int instrID;
char *filename;
{
    if (ibwrtf (bd[instrID], filename) <= 0)
        tga1240_err = 228;
    else
        tga1240_err = 0;

    return tga1240_err;
}
/*=========================================================================*/
/* Function: Check for execution errors                                    */
/* Purpose:  Interrogate the instrument for any execution errors           */
/*=========================================================================*/
int tga1240_check_exec_err(int instrumentID)
{
int nb;
	if (tga1240_err != 0)    
        return tga1240_err;
	if (tga1240_snd_cmd(instrumentID,"EER?\012") != 0)
	    return tga1240_err;
	if (tga1240_get_resp(instrumentID,cmd,&nb) != 0)
        return tga1240_err;
    Scan(cmd,"%d",&nb);
	if (nb >= 100)
        tga1240_err = nb;
    return tga1240_err;
}
/*=========================================================================*/
/* Function: Setup Arrays                                                  */
/* Purpose:  This function is called by the init routine to initialize     */
/*           global arrays.                                                */
/*=========================================================================*/
void tga1240_setup_arrays ()
{
    func[0] = "SINE";
    func[1] = "SQUARE";
    func[2] = "TRIANG";
    func[3] = "DC";
    func[4] = "POSRMP";
    func[5] = "NEGRMP";
    func[6] = "COSINE";
    func[7] = "HAVSIN";
    func[8] = "HAVCOS";
    func[9] = "SINC";
    func[10] = "PULSE";
    func[11] = "PULSTRN";
    func[12] = "ARB";
    func[13] = "SEQ";
		
	freq[0] = "WAVFREQ";
	freq[1] = "WAVPER";
	freq[2] = "CLKFREQ";
	freq[3] = "CLKPER";

	zload[0] = "OPEN";
    zload[1] = "50";
	zload[2] = "600";
	
    opsw[1] = "ON";
    opsw[0] = "OFF";
	opsw[2] = "NORMAL";
	opsw[3] = "INVERT";
	
	trig_src[0] = "INT";
	trig_src[1] = "EXT";
	trig_src[2] = "MAN";
	trig_src[3] = "PREV";
	trig_src[4] = "NEXT";

    set_status[0] = "*SRE";
    set_status[1] = "*ESE";
    set_status[2] = "*PRE";

    set_store[0] = "*RCL";
    set_store[1] = "*SAV";

    get_status[0] = "*STB?";
    get_status[1] = "*ESR?";
    get_status[2] = "*SRE?";
    get_status[3] = "*ESE?";
    get_status[4] = "*PRE?";
    get_status[5] = "EER?";

 	sync_op[0] = "AUTO";
 	sync_op[1] = "WFMSYNC";
 	sync_op[2] = "POSNMKR";
 	sync_op[3] = "BSTDONE";
 	sync_op[4] = "SEQSYNC";
 	sync_op[5] = "TRIGGER";
 	sync_op[6] = "SWPTRG";
 	sync_op[7] = "PHASLOC";

	filt[0] = "AUTO";
	filt[1] = "EL10";
	filt[2] = "EL16";
	filt[3] = "BESS";
	filt[4] = "NONE";
	
	trig_out[0] = "AUTO";
	trig_out[1] = "WFMEND";
	trig_out[2] = "POSNMKR";
	trig_out[3] = "SEQSYNC";
	trig_out[4] = "BSTDONE";
	
	hold_stat[0] = "DISAB";
	hold_stat[1] = "ENAB";
	hold_stat[2] = "ON";
	hold_stat[3] = "OFF";
	
	swp_type[0] = "CONT";
	swp_type[1] = "TRIG";
	swp_type[2] = "THLDRST";
	
	swp_dirn[0] = "UP";
	swp_dirn[1] = "DOWN";
	swp_dirn[2] = "UPDN";
	swp_dirn[3] = "DNUP";
	
	swp_spacing[0] = "LIN";
	swp_spacing[1] = "LOG";
	
	swp_sync[0] = "OFF";
	swp_sync[1] = "ON";
	
	seq_stp[0] = "COUNT";
	seq_stp[1] = "TRGEDGE";
	seq_stp[2] = "TRGLEV";
	
	mod_src[0] = "OFF";
	mod_src[1] = "EXT";
	mod_src[2] = "PREV";
	
	mod_type[0] = "AM";
	mod_type[1] = "SCM";
	
	sum_atten[0] = "0dB";
	sum_atten[1] = "10dB";
	sum_atten[2] = "20dB";
	sum_atten[3] = "30dB";
	sum_atten[4] = "40dB";
	sum_atten[5] = "50dB";
	
	lock_mode[0] = "INDEP";
	lock_mode[1] = "MASTER";
	lock_mode[2] = "FTRACK";
	lock_mode[3] = "SLAVE";
}
/*= THE END ===============================================================*/

